(in-package "CL-USER")

; This file contains the functions for type checking and type
; fixing. As a side effect, type checking puts the formula into the
; dag form that will be used for simplification. Each node in the dag
; has a slot for the type, which type checking also provides.

; There are two kinds of types: definite, and abstract. Definite types
; are positive integers, and correspond to the size of the bit vector
; returned by the formula. Abstract types are used when we do not know
; the type for sure. For example, formulas can contain integers such
; as 7. It is unclear how big a bit-vector will be used to represent
; 7. In these cases, the type is (>= . n) where n is the minimum
; number of bits needed to represent the formula. For this
; calculation, integers are assumed to be in signed
; representation. Hence, 7 has type (>= . 4).

; The the fixing functions replace integers with bit-vectors, using
; the context to figure out the definite type to give the vector. For
; example, if I have the formula (= x 7), and x is a 6 bit bit-vector,
; fix-ints will replace 7 with (const 0 0 0 1 1 1). This, in turn
; eliminates the abstract types, leaving us with only definite types.

(load "dbmc-structs")
(load "helper-functions")


;returns the type of the variable varname in typing environment e.
(defun get-var-type (varname e)
  (second (assoc varname e)))

;given a type, it returns the minimum type. That is, if tp is
;abstract, it returns the minimum possible type.
(defun mintype (tp)
  (if (atom tp)
      tp
    (cdr tp)))

;returns whether t1 and t2 are compatible types.
(defun compatible-types (t1 t2)
  (cond ((and (atom t1)
	      (atom t2))
	 (= t1 t2))         ;if they are both definite, they must be equal
	((atom t1)
	 (<= (cdr t2) t1))  ;if only one is definite, it must be greater than the mintype of the other.
	((atom t2)
	 (<= (cdr t1) t2))
	(t t)))             ;two abstract types are always compatible.

;t1 is a subtype of t2 if they are compatible and the mintype of t1 is at most that of t2.
(defun subtypeof (t1 t2)
  (and (compatible-types t1 t2)
       (<= (mintype t1) (mintype t2))))

;chooses the most restrictive of the two types.
(defun pick-type (t1 t2)
  (if (or (and (consp t1)
	       (atom t2))
	  (and (consp t1)
	       (consp t2)
	       (< (cdr t1) (cdr t2))))
      t2
    t1))

;does the same as min-type, but gives a warning if the type is
;ambiguous. This is used for top level expressions, where there is no
;context to infer the definite type of an ambiguously typed statement.
(defun check-ambiguity (tp form)
  (if (atom tp)
      tp
    (progn
      (format t "~&Warning: top-level expression is of ambiguous type. Will use smallest possible type.")
      (format t "~&type: ~a~&form: ~a." tp form)
      (cdr tp))))

;type-checks a list of forms, making sure they are all of compatible
;type. 
;a is the list being checked 
;tl is a boolean that indicates if these are top-level expressions
; (whether we should check for ambiguity) 
;db lets us know which bits of varibles are defined. This is important
; in the case of locals, where users can define variables bit by bit.
;e is the typing environment
;d is the description
(defun compatible-typelist (a tl db e d)
;  (format t "~&compatible-typelist. a: ~a" a)
  (let ((type '(>= . 0))
	(type2 nil)
	(current-arg nil)
	(formlist nil))
    (dolist (x a (values (if tl 
			     (check-ambiguity type x) 
			   type) 
			 formlist))
      (let ((f (type-check x db e d)))
	(if f
	    (progn
	      (setf formlist (append formlist
				     (list f)))
	      (setf type2 (formula-type f))
	      (if (compatible-types type type2)
		  (progn
		    (setf type (pick-type type type2))
		    (when (equal type type2) (setf current-arg x)))
		(return (values (format t
					"~&Type Error: Type mismatch: ~&~a:~a~&~a:~a"
					current-arg
					type
					x
					type2)
				nil))))
	  (return (values nil nil)))))))

;a top-level type-check. it checks for ambiguity.
;form is the form being checked.
;db lets us know which bits of varibles are defined. This is important
; in the case of locals, where users can define variables bit by bit.
;e is the typing environment
;d is the description
(defun tl-type-check (form db e d)
  (let ((tc (type-check form db e d)))
    (setf (formula-type tc) (check-ambiguity (formula-type tc) form))
    tc))
;    (when (and tc
;	       (check-ambiguity (formula-type tc) form))
;      tc)))

;certain formulas (such as tests for ifs) are required to be of type
;1. this funciton type checks form and verifies that it is of type 1.
;form is the form being checked.
;db lets us know which bits of varibles are defined. This is important
; in the case of locals, where users can define variables bit by bit.
;e is the typing environment
;d is the description
(defun type-check-1 (form db e d)
  (let ((tc (type-check form db e d)))
    (if (and tc
	     (equal (formula-type tc) 1))
	tc
      (format t "~&Type error: expecting a formula of type 1. You wrote: ~a" form))))

;returns a list of the integers between r1 and r2
(defun list-range (r1 r2)
  (do ((x r2)
       (lst nil))
      ((<= x (1- r1)) lst)
    (setf lst (cons x lst))
    (setf x (1- x))))

;define which bits are set in our defbits list, lst.
(defun set-bits (lst var bits)
  (cond ((endp lst) (list (cons var bits))) ;DARON: just changed this.
	((eq (caar lst) var)
	 (cons (cons var bits)
	       (cdr lst)))
	(t (cons (car lst) (set-bits (cdr lst) var bits)))))

;merges 2 sorted lists of bits into 1 sorted list of bits. used to add
;bits to the list of those defined for a variable.
(defun merge-bits (lst1 lst2)
  (cond ((endp lst1) lst2)
	((endp lst2) lst1)
	(t
	 (let ((i1 (car lst1))
	       (i2 (car lst2)))
	   (cond ((< i1 i2) (cons i1 (merge-bits (cdr lst1) lst2)))
		 ((> i1 i2) (cons i2 (merge-bits lst1 (cdr lst2))))
		 (t         (cons i1 (merge-bits (cdr lst1) (cdr lst2)))))))))

;modifies the list of bits defined for a given variable in the defbits list lst.
(defun add-bits (lst var bits)
  (cond ((endp lst) (list (cons var bits)))
	((eq (caar lst) var)
	 (cons (cons var (merge-bits bits (cdar lst)))
	       (cdr lst)))
	(t (cons (car lst) (add-bits (cdr lst) var bits)))))

;returns whether bit bit of variable var is defined according to defbits.
(defun bit-defined? (defbits var bit)
  (let ((v (assoc var defbits)))
    (or (not v)
	(member bit (cdr v)))))

;returns whether a range of bits is defined for variable var according
;to defbits.
(defun bit-range-defined? (defbits var b1 b2)
  (let ((v (assoc var defbits)))
    (if (not v)
	t
      (do ((v (cdr v))
	   (c b1))
	  ((<= c (1+ b2)) t) ;DARON: just changed
	;(format t "~&  c: ~a v: ~a" c v)
	(cond ((endp v) (return nil))
	      ((< (car v) c)
	       (setf v (cdr v)))
	      ((= (car v) c)
	       (setf v (cdr v))
	       (setf c (1+ c)))
	      (t (return nil)))))))

;returns whether any of the bits between b1 and b2 are defined for
;variable var according to defbits.
(defun any-bit-range-defined? (defbits var b1 b2)
  (let ((v (assoc var defbits)))
    (if (not v)
	t
      (do ((v (cdr v))
	   (c b1))
	  ((<= c (1+ b2)) nil) ;DARON: just changed
	(cond ((endp v) (return nil))
	      ((< (car v) c)
	       (setf v (cdr v)))
	      ((= (car v) c)
	       (return t))
	      (t (setf v (cdr v))
		 (1+ c)))))))

;returns whether all of variable var is defined according to defbits.
(defun var-defined? (defbits var e)
  (let ((vtp (get-var-type var e)))
    (bit-range-defined? defbits var 0 (1- vtp))))

;returns whether any bit of variable var is defined according to defbits.
(defun any-var-defined? (defbits var e)
  (let ((vtp (get-var-type var e)))
    (any-bit-range-defined? defbits var 0 (1- vtp))))

;removes a var from defbits.
(defun remove-from-defbits (defbits var)
  (cond ((endp defbits)
	 nil)
	((eq (caar defbits) var)
	 (remove-from-defbits (cdr defbits) var))
	(t
	 (cons (car defbits)
	       (remove-from-defbits (cdr defbits) var)))))

;type-checks local bindings.
;bindings is the list of bindings to be checked.
;vars is the list of variables to be bound bit-by-bit.
;e is the typing environment.
;d is the desc.
(defun local-binding-type-check (bindings vars defbits e d)
  ;(format t "~&local-binding-type-check: defbits=~a" defbits)
  (let ((bforms nil))
    (dolist (b bindings (values defbits e bforms))
      ;(format t "~&  b=~a" b)
      (let ((v (car b))
	    (len (length b)))
	(cond ((= len 3)
	       (let ((tc (type-check (nth 2 b) defbits e d))
		     (tp (nth 1 b)))
		 (cond  ((not tc) (return nil))
			((subtypeof (formula-type tc)
				    tp)
			 (setf bforms (app-item bforms
						(make-formula :fn 'new-binding
							      :type tp
							      :args (list v tc))))
			 (setf defbits (remove-from-defbits defbits v))
			 (setf e (cons (list v (nth 1 b)) e)))
			(t (return (format t "~&Type error: Incompatible types in binding: ~a" b))))))
	      ((and (atom v)
		    (member v vars))
	       (let ((tc (type-check (nth 1 b) defbits e d))
		     (tp (nth 1 (assoc v e))))
		 (cond  ((not tc) (return nil))
			((any-var-defined? defbits v e)
			 (return (format t "~&Attempt to redefine ~a: ~a" v b)))
			((subtypeof (formula-type tc)
				    tp)
			 (setf bforms (app-item bforms
						(make-formula :fn 'var-binding
							      :type tp
							      :args (list v tc))))
			 (setf defbits (set-bits defbits v (list-range 0 (1- (get-var-type v e))))))
			(t (return (format t "~&Type error: Incompatible types in binding: ~a" b))))))
	      ((atom v)
	       (let ((tc (tl-type-check (nth 1 b) defbits e d)))
		 (cond ((not tc) (return nil))
		       (t
			(setf bforms (app-item bforms 
					       (make-formula :fn 'new-binding
							     :type (formula-type tc)
							     :args (list v tc))))
			(setf defbits (remove-from-defbits defbits v))
			(setf e (cons (list v (formula-type tc)) e))))))
	      ((and (= (length v) 2)
		    (integerp (nth 1 v)))
	       (let ((tc (type-check (nth 1 b) defbits e d))
		     (bit (nth 1 v))
		     (vtp (get-var-type (car v) e)))
		 (cond  ((not tc) (return nil))
			((bit-defined? defbits (car v) bit)
			 (return (format t "~&Attempt to redefine ~a: ~a" v b)))
			((not (< (nth 1 v)
				 vtp))
			 (return (format t "~&Type error: bit index out of bounds in binding: ~a" b)))
			((subtypeof (formula-type tc)
				    1)
			 (setf bforms (app-item bforms
						(make-formula :fn 'bit-binding
							      :type 1
							      :args (list v tc))))
			 (setf defbits (add-bits defbits (car v) (list bit))))
			(t (return (format t "~&Type error: Incompatible types in binding: ~a" b))))))
	      ((and (= (length v) 3)
		    (integerp (nth 1 v)))
	       (let* ((tc (type-check (nth 1 b) defbits e d))
		      (b1 (nth 1 v))
		      (b2 (nth 2 v))
		      (tp (+ (- b2 b1) 1))
		      (vtp (get-var-type (car v) e)))
		 (cond ((not tc) (return nil))
		       ((not (< b1 b2))
			(return (format t 
					"~&Type error: second bit number in bit range must be greater than the first: ~a"
					b)))
		       ((not (< b2 vtp))
			(return (format t "~&Type error: bit index out of bounds in binding: ~a" b)))
		       ((any-bit-range-defined? defbits (car v) b1 b2)
			(return (format t "~&Attempt to redefine ~a: ~a" v b)))
		       ((subtypeof (formula-type tc)
				   tp)
			(setf bforms (app-item bforms
					       (make-formula :fn 'bits-binding
							     :type tp
							     :args (list v tc))))
			(setf defbits (add-bits defbits (car v) (list-range b1 b2))))
		       (t (return (format t "~&Type error: Incompatible types in binding: ~a" b))))))
	      (t
	       (let ((tc (type-check (nth 1 b) defbits e d)))
		 (if tc
		     (let* ((tsum 0)
			    (bforms2 nil)
			    (cont (dolist (x v t)
				    ;(format t "~&x: ~a" x)
				    (cond ((and (atom x)
						(member x vars))
					   (let ((tp (nth 1 (assoc x e))))
					     (cond ((any-var-defined? defbits x e)
						    (return (format t "~&Attempt to redefine ~a: ~a" x b)))
						   (t
						    (setf bforms2 (app-item bforms2
									    (make-formula :fn 'var-mv-binding
											  :type tp
											  :args (list x))))
						    (setf defbits 
							  (set-bits defbits x (list-range 0 (1- (get-var-type x e)))))
						    (setf tsum (+ tsum tp))))))
					  ((atom x)
					   (setf bforms2 (app-item bforms2 
								   (make-formula :fn 'new-mv-binding
										 :type 1
										 :args (list x))))
					   (setf tsum (+ tsum 1))
					   (setf defbits (remove-from-defbits defbits x))
					   (setf e (cons (list x 1) e)))
					  ((and (= (length x) 2)
						(member (car x) vars))
					   (let ((tp (get-var-type (car x) e))
						 (bit (nth 1 x)))
					     (cond ((>= (nth 1 x) tp)
						    (return (format t "~&Type error: bit index out of bounds in binding: ~a" x)))
						   ((bit-defined? defbits (car x) (nth 1 x))
						    (return (format t "~&Attempt to redefine ~a: ~a" x b)))
						   (t
						    (setf bforms2 (app-item bforms2
									    (make-formula :fn 'bit-mv-binding
											  :type 1
											  :args x)))
						    (setf defbits (add-bits defbits (car x) (list bit)))
						    (setf tsum (+ tsum 1))))))
					  ((= (length x) 2)
					   (let ((tp (nth 1 x)))
					     (setf bforms2 (app-item bforms2
								     (make-formula :fn 'new-mv-binding
										   :type tp
										   :args (list (car x)))))
					     (setf tsum (+ tsum tp))
					     (setf defbits (remove-from-defbits defbits v))
					     (setf e (cons x e))))
					  (t
					   (let* ((var (car x))
						  (b1 (nth 1 x))
						  (b2 (nth 2 x))
						  (tp (+ (- b2 b1) 1))
						  (vtp (nth 1 (assoc var e))))
					     (cond ((<= b2 b1)
						    (return (format t 
								    "~&Type error: second bit number in bit range must be greater than the first: ~a"
								    x)))
						   ((<= vtp b2)
						    (return (format t "~&Type error: bit index out of bounds in binding: ~a" x)))
						   ((any-bit-range-defined? defbits (car x) b1 b2)
						    (return (format t "~&Attempt to redefine ~a: ~a" x b)))
						   (t 
						    (setf bforms2 (app-item bforms2
									    (make-formula :fn 'bits-mv-binding
											  :type tp
											  :args x)))
						    (setf defbits (add-bits defbits (car x) (list-range b1 b2)))
						    (setf tsum (+ tsum tp))))))))))
		       (if cont
			   (if (subtypeof (formula-type tc)
					  tsum)
			       (setf bforms (app-item bforms
						      (make-formula :fn 'mv-binding
								    :type tsum
								    :args (list bforms2 tc))))
			     (return (format t "~&Type error: Incompatible types in binding: ~a" b)))
			 (return nil)))
		   (return nil)))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; added by roma on 9 march - adding a function
; string-list to convert string to list - used in binary
; changed to implement Verilog representation - 18 May

(defun string-list (form)
  (setq cform (subseq (string-downcase form) 2 (length (string form))))
  (setf s-list '(0))
  (dotimes (x (length cform) (subseq s-list 1 (length s-list)))
    (if (char-equal (elt cform x) #\1)
	(setf s-list (append s-list '(1)))
      (setf s-list (append s-list '(0))))))

(defun string-list-hex (form)
  (setq cform (subseq (string-downcase form) 2 (length (string form))))
  (setf s-list '(0))
  (dotimes (x (length cform) (subseq s-list 1 (length s-list)))
    (if (and (char>= (elt cform x) #\0)
	     (char<= (elt cform x) #\9))
	(setf s-list (append s-list (bin-spec-list (- (char-int (elt cform x)) 48) 4)))
      (setf s-list (append s-list (bin-spec-list (- (char-int (elt cform x)) 87) 4))))))

(defun n-ubv-0 (n)
  (cond ((zp n)
         '(0))
        ((evenp n)
         (append (n-ubv (/ n 2)) '(0)))
        (t (append (n-ubv (/ (1- n) 2)) '(1)))))

(defun un-ubv (form)
  (n-ubv-0 (read-from-string (subseq (string form) 0 (1- (length (string form)))))))
  
(defun string-list-octal (form)
  (setq cform (subseq (string-downcase form) 2 (length (string form))))
  (setf s-list '(0))
  (dotimes (x (length cform) (subseq s-list 1 (length s-list)))
    (setf s-list (append s-list (bin-spec-list (- (char-int (elt cform x)) 48) 3)))))
      
; depending on the length specified return binary number
(defun bin-spec-list (form size)
  (setq s-list (n-ubv form))
  (setq n-len (length s-list))
  (if (= form 0)
      (dotimes (x size s-list)
	(setq s-list (append '(0) s-list))) 
    (dotimes (x (- size n-len) s-list)
	(setq s-list (append '(0) s-list)))))

;the main type-checking function.
;form is the form being checked.
;db tells us which bits are defined for variables that have been
; defined bit-by-bit
;e is the typing environment
;d is the desc
(defun type-check (form db e d)
  ;(format t "~&type-check: ~&form=~a" form)
  (let ((cform form)) ;(tl-replace-constants form d)))
    (cond ((integerp cform)
	   (make-formula :fn 'int
			 :type (cons '>= (size cform))
			 :args cform))
	  ((atom cform)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; changed by roma on 9 March to add binary nos to const

	   (cond ((and (char-equal (elt (string cform) 0) #\0)
		       (char-equal (elt (string cform) 1) #\b))
		  (format t "~& Binary number: ~a" cform)
		  (make-formula :fn 'const
				:type (- (length (string cform)) 2)
				:args (string-list cform)))
		 ((and (char-equal (elt (string cform) 0) #\0)
		       (char-equal (elt (string cform) 1) #\h))
		  (format t "~& Hexadecimal number: ~a" cform)
		  (make-formula :fn 'const
				:type (* (- (length (string cform)) 2) 4)
				:args (string-list-hex cform)))
		 ((and (char-equal (elt (string cform) 0) #\0)
		       (char-equal (elt (string cform) 1) #\o))
		  (format t "~& Octal number: ~a" cform)
		  (make-formula :fn 'const
				:type (* (- (length (string cform)) 2) 3)
				:args (string-list-octal cform)))
		 ((char-equal (elt (string cform) (- (length (string cform)) 1)) #\u)
		  (format t "~& Unsigned: ~a" cform)
		  (make-formula :fn 'const
				:type (list-length (un-ubv cform))
				:args (un-ubv cform)))
	         ((not (var-defined? db cform e)) 
                  (format t "~&Attempt to use variable that is not fully defined: ~a" cform))
                 ((assoc cform (desc-defs d))
                  (make-formula :fn 'def
                                :type (get-var-type cform e)
                                :args (list cform)))
                 (t (make-formula :fn 'var
                                  :type (get-var-type cform e)
                                  :args (list cform)))))
	  (t (let ((fn (car cform))
		   (args (cdr cform)))
	       (cond ((isvar fn e)
		      (let* ((tp (get-var-type fn e))
			     (vform (if (assoc cform (desc-defs d))
                                        (make-formula :fn 'def
                                                      :type tp
                                                      :args (list fn))
                                      (make-formula :fn 'var
                                                    :type tp
                                                    :args (list fn)))))
			(if (< (car (last args)) tp)
			    (if (= (length args) 1)
				(if (bit-defined? db fn (car args))
				    (make-formula :fn 'bit
						  :type 1
						  :args (cons vform args))
				  (format t "~&Attempt to use undefined bit: ~a" cform))
			      (if (bit-range-defined? db fn (car args) (nth 1 args))
				  (make-formula :fn 'bits
						:type (+ (- (nth 1 args) (nth 0 args)) 1)
						:args (cons vform args))
				(format t "~&Attempt to use undefined bits: ~a" cform)))
			  (format t
				  "~&Type error: Index out of bounds: ~a"
				  form))))
		     ((or (eq fn 'AF)
			  (eq fn 'AG))
		      (let ((fm (type-check (car args) db e d)))
			(when fm
			  (make-formula :fn fn
					:type 1
					:args (list fm)))))
		     ((or (eq fn '<->)
			  (eq fn 'and)
			  (eq fn 'or)
			  (eq fn 'xor))
		      (multiple-value-bind
			(tp a)
			(compatible-typelist args nil db e d)
			(when tp
			  (make-formula :fn fn
					:type tp
					:args a))))
		     ((or (eq fn '+)
			  (eq fn '-))
		      (multiple-value-bind
			(tp a)
			(compatible-typelist args nil db e d)
			(when tp
			  (make-formula :fn fn
					:type (if (atom tp)
						  (+ tp 
						     (ceil-log (length a)))
						(cons '>= (+ (cdr tp) 1)))
					:args a))))
		   
		     ((eq fn '=)
		      (multiple-value-bind
			(tp a)
			(compatible-typelist args nil db e d)
			(when tp
			  (make-formula :fn fn
					:type 1
					:args a))))
		     ((eq fn 'cond)
		      (let ((conds (mapcar #'car args))
			    (exps (mapcar #'cadr args))
			    (args nil))
			(multiple-value-bind
			  (tp a)
			  (compatible-typelist exps nil db e d)
			  (when tp
			    (dolist (c conds (make-formula :fn fn
							   :type tp
							   :args args))
			      (let ((tc (type-check-1 c db e d)))
				(if tc
				    (setf args (append args
						       (list (list tc (car a)))))
				  (return nil)))
			      (setf a (cdr a)))))))
		     ((eq fn 'local)
		      (let* ((len (length args))
			     (a1 (if (= len 3) (fix-var-list (nth 0 args) d) nil))
			     (vars (mapcar #'car a1))
			     (a2 (if (= len 3) (nth 1 args) (nth 0 args)))
			     (a3 (if (= len 3) (nth 2 args) (nth 1 args))))
			(setf e (append a1 e))
			(dolist (a a1 t)
			  (setf db (cons (list (car a)) db)))
			(multiple-value-bind
			  (db e bforms)
			  (local-binding-type-check a2 vars db e d)
			  (when e
			    (let ((tc (type-check a3 db e d)))
			      (when tc
				(make-formula :fn 'local
					      :type (formula-type tc)
					      :args (list a1 bforms tc))))))))
		     ((eq fn 'if)
		      (let ((testf (type-check-1 (nth 0 args) db e d)))
			(when testf
			  (let ((thenf (type-check (nth 1 args) db e d)))
			    (when thenf
			      (let ((elsef (type-check (nth 2 args) db e d)))
				(when elsef
				  (if (compatible-types (formula-type thenf)
							(formula-type elsef))
				      (make-formula :fn fn
						    :type (pick-type (formula-type thenf)
								     (formula-type elsef))
						    :args (list testf thenf elsef))
				    (format t
					    "~&Type Error: Incompatible then and else clauses: ~&~a"
					    form)))))))))
		     ((eq fn 'cat)
		      (let ((sum 0)
			    (newargs nil))
			(dolist (x args (make-formula :fn fn
						      :type sum
						      :args newargs))
			  (let ((curform (tl-type-check x db e d)))
			    (if curform
				(progn
				  (setf sum (+ sum (formula-type curform)))
				  (setf newargs (append newargs (list curform))))
			      (return nil))))))
		     ((eq fn '->)
		      (let ((f0 (type-check-1 (nth 0 args) db e d)))
			(when f0
			  (let ((f1 (type-check-1 (nth 1 args) db e d)))
			    (when f1
			      (make-formula :fn fn
					    :type 1
					    :args (list f0 f1)))))))
		     ((or (eq fn '>>)
			  (eq fn '<<))
		      (let ((f0 (type-check (nth 0 args) db e d)))
			(when f0
			  (make-formula :fn fn
					:type (formula-type f0)
					:args (list f0 (nth 1 args))))))
		     ((or (eq fn 'not)
			  (eq fn 'inc)
			  (eq fn 'dec)
			  (eq fn 'next))
		      (let ((f (type-check (nth 0 args) db e d)))
			(when f
			  (make-formula :fn fn
					:type (formula-type f)
					:args (list f)))))
		     ((or (eq fn 'foldl)
			  (eq fn 'foldr))
		      (let* ((op (car args))
			     (exp (nth 1 args))
			     (funct (get-funct op d)))
			(if (and funct
				 (not (and (= (length (funct-params funct)) 2)
					   (dolist (p (funct-params funct) t)
					     (when (not (= (nth 1 p) 1)) (return nil))))))
			    (format t 
				    "~&Type Error: Function passed to ~a must be a binary function with parameters of type 1. You wrote ~a"
				    fn
				    form)
			  (let ((f (type-check exp db e d)))
			    (when f
			      (make-formula :fn fn
					    :type 1
					    :args (list op f)))))))
		     ((eq fn 'type)
		      (make-formula :fn fn
				    :type (nth 1 args)
				    :args (nth 0 args)))
		     ((eq fn 'bit)
		      (let ((f (tl-type-check (nth 0 args) db e d)))
			(when f
			  (if (< (nth 1 args)
				 (mintype (formula-type f)))
			      (make-formula :fn fn
					    :type 1
					    :args (cons f (cdr args)))
			    (format t
				    "~&Type Error: Bit number out of bounds: ~a"
				    form)))))
		     ((eq fn 'bits)
		      (let ((f (tl-type-check (nth 0 args) db e d)))
			(when f
			  (if (< (nth 2 args)
				 (mintype (formula-type f)))
			      (make-formula :fn fn
					    :type (+ (- (nth 2 args)
							(nth 1 args))
						     1)
					    :args (cons f (cdr args)))
			    (format t
				    "~&Type Error: Bit number out of bounds: ~a"
				    form)))))
		     ((eq fn 'ext)
		      (let ((f (type-check (nth 0 args) db e d)))
			(when f
			  (if (< (mintype (formula-type f))
				 (nth 1 args))
			      (make-formula :fn fn
					    :type (nth 1 args)
					    :args (list f))
			    (format t
				    "~&Type Error: Extension type too small: ~a"
				    form)))))
		     ((eq fn 'const)
		      (make-formula :fn fn
				    :type (length args)
				    :args args))
		     (t
		      (let ((f (get-funct fn d)))
			(when f
			  (let* ((type (funct-type f))
				 (params (funct-params f))
				 (len (length params))
				 (len2 (length args))
				 (fret (make-formula :fn fn
						     :type type
						     :args nil)))
			    (if (= len len2)
				(dolist (p params fret)
				  (let ((ptype (nth 1 p))
					(aform (type-check (nth 0 args) db e d)))
				    (if aform
					(if (subtypeof (formula-type aform) ptype)
					    (setf (formula-args fret)
						  (append (formula-args fret)
							  (list aform)))
					  (return (format t
							  "~&Type error: Type mismatch: Attempt to pass ~a of type ~a as type ~a"
							  (nth 0 args)
							  (formula-type aform)
							  ptype)))
				      (return nil)))
				  (setf args (cdr args)))
			      (format t
				      "~&Type error: ~a expects ~a arguments. You gave it ~a: ~&~a"
				      fn
				      len
				      len2
				      form))))))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; int fixing                        ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;fixes a list of formulas.
;lst is the list being fixed
;type is the type to give the elements of the list
;d is the desc
(defun fix-ints-list (lst type d)
  (let ((f #'(lambda (form) (fix-ints form type d))))
    (mapcar f lst)))

;top-level fixes ints in a list.
(defun tl-fix-ints-list (lst d)
  (let ((f #'(lambda (x) (tl-fix-ints x d))))
    (mapcar f lst)))

;finds the most restrictive type in a list. this is the type we will
;give everything in the list.
(defun list-type (lst)
  (let ((tp 0))
    (dolist (x lst t)
      (let ((xtp (formula-type x)))
	(if (atom xtp)
	    (return (setf tp xtp))
	  (if (>= (mintype xtp) (mintype tp))
	      (setf tp xtp)))))
    (mintype tp)))

;converts an integer into the corresponding signed bit-vector of the given type.
(defun make-int-type (i type)
  (let ((s (size i)))
    (if (< type s)
	(nthcdr (- s type) (i-sbv i))
      (sign-extend (i-sbv i) type))))

;the main fixign algorithm.
(defun fix-ints (form type d)
  ;(format t "~&fix-ints. form=~a" form)
  (let ((fn (formula-fn form))
	(args (formula-args form)))
    (cond ((or (eq fn 'int)
	       (eq fn 'type))
	   (make-formula :fn 'const
			 :type type
			 :args (make-int-type (formula-args form) type)))
	  ((or (eq fn 'AG)
	       (eq fn 'AF))
	   (make-formula :fn fn
			 :type type
			 :args (tl-fix-ints-list args d)))
	  ((or (eq fn '<->)
	       (eq fn 'and)
	       (eq fn 'or)
	       (eq fn 'xor)
	       (eq fn 'not)
	       (eq fn 'inc)
	       (eq fn 'dec)
	       (eq fn 'next))
	   (make-formula :fn fn
			 :type type
			 :args (fix-ints-list args type d)))
	  ((or (eq fn 'foldl)
	       (eq fn 'foldr))
	   (make-formula :fn fn
			 :type type
			 :args (list (car args) (tl-fix-ints (nth 1 args) d))))
	  ((or (eq fn 'local))
	   (let ((bindings (nth 1 args))
		 (bforms nil))
	     (dolist (b bindings (make-formula :fn fn
					       :type type
					       :args (list (car args)
							   bforms
							   (fix-ints (nth 2 args) type d))))
	       (let ((bfn (formula-fn b))
		     (btp (formula-type b))
		     (bargs (formula-args b)))
	       (if (eq bfn 'dec-binding)
		   (setf bforms (app-item bforms b))
		 (let ((exp (car (last bargs))))
		   (setf bforms 
			 (app-item bforms
				   (make-formula :fn bfn
						 :type btp
						 :args (app-item (butlast bargs 1)
								 (fix-ints exp btp d)))))))))))
	  ((eq fn 'binding-dec)
	   (make-formula :fn fn
			 :type type
			 :args (car args)))
	  ((eq fn 'binding-def)
	   (make-formula :fn fn
			 :type type
			 :args (list (car args)
				     (fix-ints-list (cdr args) type d))))
	  ((or (eq fn '+)
	       (eq fn '-))
	   (make-formula :fn fn
			 :type type
			 :args (fix-ints-list args 
					      (- type (ceil-log (length args)))
					      d)))
	  ((eq fn '=)
	   (let* ((tp (list-type args))
		  (f #'(lambda (form) (fix-ints form tp d))))
	     (make-formula :fn fn
			   :type type
			   :args (mapcar f args))))
	  ((eq fn 'cond)
	   (let ((f #'(lambda (arg) (list (tl-fix-ints (nth 0 arg) d)
					  (fix-ints (nth 1 arg) type d)))))
	     (make-formula :fn fn
			   :type type
			   :args (mapcar f args))))
	  ((eq fn 'if)
	   (make-formula :fn fn
			 :type type
			 :args (cons (tl-fix-ints (car args) d)
				     (fix-ints-list (cdr args) type d))))
	  ((or (eq fn '->)
	       (eq fn 'cat))
	   (make-formula :fn fn
			 :type type
			 :args (tl-fix-ints-list args d)))
	  ((or (eq fn '>>)
	       (eq fn '<<))
	   (make-formula :fn fn
			 :type type
			 :args (list (fix-ints (nth 0 args) type d)
				     (nth 1 args))))
	  ((or (eq fn 'bit)
	       (eq fn 'bits))
	   (make-formula :fn fn
			 :type type
			 :args (cons (tl-fix-ints (car args) d)
				     (cdr args))))
	  ((eq fn 'ext)
	   (make-formula :fn fn
			 :type type
			 :args (tl-fix-ints-list args d)))
	  ((get-funct fn d)
	   (let* ((f (get-funct fn d))
		  (params (funct-params f))
		  (newargs nil))
	     (dolist (p params t)
	       (setf newargs (append newargs
				     (list (fix-ints (car args) (nth 1 p) d))))
	       (setf args (cdr args)))
	     (make-formula :fn fn
			   :type type
			   :args newargs)))
	  (t form))))
	       
;top-level fixing. it chooses the most restrictive type possible for
;form.
(defun tl-fix-ints (form d)
  (fix-ints form (mintype (formula-type form)) d))

;fix a function funct in desc d.
(defun fix-function (funct d)
  (let* ((fn (cdr funct))
	 (tcf (tl-type-check (replace-constants (funct-body fn) d)
			     nil
			     (funct-params fn)
			     d)))
    (when tcf
      (cond ((subtypeof (formula-type tcf)
			(funct-type fn))
	     (setf (funct-body fn)
		   (fix-ints tcf (funct-type fn) d)))
	    (t (format t "~&Function ~a: function type ~a does not match body type: ~a"
		       (car funct)
		       fn
		       (funct-body fn)))))))

;fix the defs section in d.
(defun fix-defs (d)
  (let ((defs (desc-defs d))
        (ndefs nil)
        (e (desc-vars d)))
    (dolist (def defs (progn (setf (desc-defs d) (reverse ndefs)) t))
      (let ((ndef (tl-type-check (replace-constants (second def) d)
                                 nil
                                 e
                                 d)))
        (cond (ndef
               (setf ndef (tl-fix-ints ndef d))
               (setf ndefs (cons (cons (first def) ndef) ndefs))
               (setf e (cons `(,(first def) ,(formula-type ndef)) e)))
              (t (return nil)))))))

;auxilary function for desc-tl-env
(defun desc-tl-env-aux (defs env)
  (if (consp defs)
      (desc-tl-env-aux (cdr defs) 
                       (cons `(,(caar defs) ,(formula-type (cdar defs))) env))
    env))

;turns the vars and defs of d into a typing environment.
(defun desc-tl-env (d)
  (desc-tl-env-aux (reverse (desc-defs d)) (desc-vars d)))

;fixes the init, trans, or spec section of d as specified.
(defun fix-section (d sect)
  (let ((e (desc-tl-env d)))
    (case sect
      (init  (let ((s (type-check-1 (replace-constants (desc-init d) d) nil e d)))
               (when s 
                 (setf (desc-init d) (tl-fix-ints s d)))))
      (trans (let ((s (type-check-1 (replace-constants (desc-trans d) d) nil e d)))
	       (when s 
		 (setf (desc-trans d) (tl-fix-ints s d)))))
      (spec  (let ((s (type-check-1 (replace-constants (desc-spec d) d) nil e d)))
               (when s 
                 (setf (desc-spec d) (tl-fix-ints s d))))))
    t))
